package jp.co.conet;

import android.app.Activity;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.content.Intent;
import android.os.Bundle;
import android.os.Handler;
import android.util.Log;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.widget.TextView;
import android.widget.Toast;

import com.neurosky.android.NeuroData;
import com.neurosky.android.NeuroRawData;
import com.neurosky.android.NeuroSky;
import com.neurosky.android.NeuroStreamParser;

public class SampleActivity extends Activity {
	
	/**   デバッグ関連   */
	private static final String LOGGER_TAG				= "NEURO_SKY";
	private static final boolean LOGGER_ENABLE		= false;
	
	/**   Intent リクエストコード   */
	private static final int REQUEST_CONNECT_DEVICE	= 0x01;
	private static final int REQUEST_ENABLE_BT				= 0x02;
	
	/**   RAW データ要否フラグ   */
	private static final boolean RAW_DATA_ENABLE = false;
	
	/**   メンバ変数   */
	private TextView digestView;
	private TextView rawView;
	private BluetoothAdapter btAdapter;
	private NeuroSky neuroSky;
	private final Handler btHandler = new Handler() {
		@Override
		public void handleMessage(android.os.Message msg) {
			switch (msg.what) {
			
				case (NeuroSky.MESSAGE_BT_DEVICE_NAME) :
					if (LOGGER_ENABLE) Log.d(LOGGER_TAG, "[ MESSAGE_BT_DEVICE_NAME ]");
					break;
				
				case (NeuroSky.MESSAGE_BT_STATUS_CHANGE) :
					if (LOGGER_ENABLE) Log.d(LOGGER_TAG, "[ MESSAGE_BT_STATUS_CHANGE ]");
				    //   変更されたステータス
				    int changedStatus = msg.arg1;
				    if (changedStatus == NeuroSky.BT_STATE_NONE) {
				    	Log.i(LOGGER_TAG, "BT_STATE_NONE");
				    } else if (changedStatus == NeuroSky.BT_STATE_CONNECTING) {
				    	Log.i(LOGGER_TAG, "BT_STATE_CONNECTING");
				    } else if (changedStatus == NeuroSky.BT_STATE_CONNECTED) {
				    	Log.i(LOGGER_TAG, "BT_STATE_CONNECTED");				    	
				    }
					break;
				
				case (NeuroSky.MESSAGE_BT_TOAST) :
					if (LOGGER_ENABLE) Log.d(LOGGER_TAG, "[ MESSAGE_BT_TOAST ]");
				    Toast.makeText(getApplicationContext(), msg.getData().getString("toast"), Toast.LENGTH_SHORT).show();
					break;
				
				case (NeuroStreamParser.MESSAGE_READ_DIGEST_DATA_PACKET) :
					if (LOGGER_ENABLE) Log.d(LOGGER_TAG, "[ MESSAGE_READ_DIGEST_DATA_PACKET ]");
				    NeuroData data = (NeuroData) msg.obj;
				    //Log.i(LOGGER_TAG, "All values : " + data.getAllString());
				    digestView.setText(data.getAllString());
					break;
				
				case (NeuroStreamParser.MESSAGE_READ_RAW_DATA_PACKET) :
					if (LOGGER_ENABLE) Log.d(LOGGER_TAG, "[ MESSAGE_READ_RAW_DATA_PACKET ]");
				    NeuroRawData rawData = (NeuroRawData) msg.obj;
				    //Log.i(LOGGER_TAG, "Raw values : " + rawData.getRawWaveValue());
				    if (RAW_DATA_ENABLE) {
				    	rawView.setText(rawData.getRawWaveValue());
				    }
					break;
			}
		};
	};
	
	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		
		if (LOGGER_ENABLE) Log.d(LOGGER_TAG, "_/_/_/_/_/   ON-CREATE   _/_/_/_/_/");
        setContentView(R.layout.main);
        digestView = (TextView) findViewById(R.id.digest_view);
        if (RAW_DATA_ENABLE) {
        	rawView = (TextView) findViewById(R.id.raw_view);
        }
        
		btAdapter = BluetoothAdapter.getDefaultAdapter();
	}

	@Override
	protected void onStart() {
		super.onStart();

		if (LOGGER_ENABLE) Log.d(LOGGER_TAG, "_/_/_/_/_/   ON-START   _/_/_/_/_/");
		if (btAdapter.isEnabled()) {
			neuroSky = new NeuroSky(btAdapter, btHandler);
		} else {
			Intent intent = new Intent(BluetoothAdapter.ACTION_REQUEST_ENABLE);
			startActivityForResult(intent, REQUEST_ENABLE_BT);
		}
	}
	
	@Override
	protected void onResume() {
		super.onResume();

		if (LOGGER_ENABLE) Log.d(LOGGER_TAG, "_/_/_/_/_/   ON-RESUME   _/_/_/_/_/");
		
		if ((neuroSky != null) && (neuroSky.getStatus() == NeuroSky.BT_STATE_CONNECTED)) {
			neuroSky.start(RAW_DATA_ENABLE);
		}			
	}
	
	@Override
	protected void onPause() {
		super.onPause();

		if (LOGGER_ENABLE) Log.d(LOGGER_TAG, "_/_/_/_/_/   ON-PAUSE   _/_/_/_/_/");

		if ((neuroSky != null) && (neuroSky.getStatus() == NeuroSky.BT_STATE_CONNECTED)) {
			neuroSky.stop();
		}
		
	}
	
	@Override
	protected void onStop() {
		super.onStop();

		if (LOGGER_ENABLE) Log.d(LOGGER_TAG, "_/_/_/_/_/   ON-STOP   _/_/_/_/_/");
		
		if (neuroSky != null) neuroSky.stop();
	}
	
	@Override
	protected void onDestroy() {
		super.onDestroy();

		if (LOGGER_ENABLE) Log.d(LOGGER_TAG, "_/_/_/_/_/   ON-DESTROY   _/_/_/_/_/");
		
		if (neuroSky != null) neuroSky.close();
	}
	
	@Override
	public boolean onCreateOptionsMenu(android.view.Menu menu) {
		MenuInflater inflater = getMenuInflater();
		inflater.inflate(R.menu.option_menu, menu);
		return true;
	}
	
	@Override
	public boolean onOptionsItemSelected(MenuItem item) {
		boolean result = false;
		switch (item.getItemId()) {
		
			case (R.id.scan) :
				Intent intent = new Intent(this, SampleDeviceListActivity.class);
			    startActivityForResult(intent, REQUEST_CONNECT_DEVICE);
			    result = true;
				break;
				
			case (R.id.clear) :
				if (neuroSky != null) {
					neuroSky.stop();
					neuroSky.close();
				}
			    result = true;
			    break;
			case (R.id.start) :
				if (neuroSky != null) {
			    	// データ受信開始
			    	neuroSky.start(RAW_DATA_ENABLE);
				}
			    result = true;
			    break;
			case (R.id.stop) :
				if (neuroSky != null) {
			    	// データ受信終了
			    	neuroSky.stop();
				}
			    result = true;
			    break;
				
		}
		return result;
	}
	
	@Override
	protected void onActivityResult(int requestCode, int resultCode, Intent data) {
		super.onActivityResult(requestCode, resultCode, data);

		if (LOGGER_ENABLE) Log.d(LOGGER_TAG, "onActivityResult : " + resultCode);
		switch (requestCode) {
		
			case (REQUEST_CONNECT_DEVICE) :
				if (resultCode == Activity.RESULT_OK) {
					String deviceAddress = data.getExtras().getString(SampleDeviceListActivity.EXTRA_DEVICE_ADDRESS);
					BluetoothDevice device = btAdapter.getRemoteDevice(deviceAddress);
					//   対象デバイスと接続
					neuroSky.connect(device);
				}
				break;
			
			case (REQUEST_ENABLE_BT) :
				if (resultCode == Activity.RESULT_OK) {
					neuroSky = new NeuroSky(btAdapter, btHandler);
				} else {
					Toast.makeText(this, R.string.err_bluetooth_not_enabled, Toast.LENGTH_SHORT).show();
				}
				break;
			
		}
	}
}
